<?php
/*
Plugin Name: Decision Tree Plugin v10
Description: Manage decision trees as JSON via a WP Admin page and render them as a quiz with Vue.js. Includes server-side + client-side JSON validation and a Start Over button.
Version: 2.1.1
Author: Kerim Hmaied
*/
if (!defined('ABSPATH')) exit;

function dtv10_enqueue_assets() {
    wp_enqueue_style('dtv10-style', plugin_dir_url(__FILE__) . 'css/decision-tree.css', [], '1.1');
    wp_enqueue_script('vue', 'https://cdn.jsdelivr.net/npm/vue@2.6.14/dist/vue.js', [], null, true);
    wp_enqueue_script('dtv10-js', plugin_dir_url(__FILE__) . 'js/decision-tree.js', ['vue'], '1.1', true);
}
add_action('wp_enqueue_scripts', 'dtv10_enqueue_assets');

add_action('admin_enqueue_scripts', function($hook){
    if ($hook === 'toplevel_page_decision-tree-v10-admin') {
        wp_enqueue_script('dtv10-admin-lint', plugin_dir_url(__FILE__) . 'js/dt-admin-lint.js', [], '1.0', true);
    }
});

function dtv10_shortcode($atts) {
    $atts = shortcode_atts(['tree' => 'energieberater'], $atts, 'decision_tree_v10');
    $tree = sanitize_file_name($atts['tree']);
    $tree_file = plugin_dir_path(__FILE__) . 'trees/' . $tree . '.json';
    if (!file_exists($tree_file)) {
        return '<p style="color:red;">Decision tree not found: ' . esc_html($tree) . '</p>';
    }
    $tree_json = file_get_contents($tree_file);
    return '<div id="decision-tree"><decision-tree></decision-tree></div>'
         . '<script id="decision-tree-data" type="application/json">' . $tree_json . '</script>';
}
add_shortcode('decision_tree_v10', 'dtv10_shortcode');

function dtv10_validate_tree_array($node, array &$errors, $path = 'root') {
    if (!is_array($node)) { $errors[] = "$path: node must be object"; return; }
    $hasQ = array_key_exists('question', $node);
    $hasR = array_key_exists('recommendation', $node);
    if ($hasQ) {
        if (!is_string($node['question']) || trim($node['question'])==='') $errors[]="$path.question invalid";
        if (!isset($node['options'])||!is_array($node['options'])||!count($node['options'])){$errors[]="$path.options empty";}
        else foreach($node['options'] as $i=>$opt){
            $p="$path.options[$i]"; if(!is_array($opt)){$errors[]="$p must be object";continue;}
            if(!isset($opt['text'])||!is_string($opt['text'])||trim($opt['text'])==='')$errors[]="$p.text invalid";
            $hasNext=array_key_exists('next',$opt);$hasRec=array_key_exists('recommendation',$opt);
            if($hasNext===$hasRec){$errors[]="$p must have next or recommendation";continue;}
            if($hasNext){dtv10_validate_tree_array($opt['next'],$errors,"$p.next");}
            else{if(!is_string($opt['recommendation'])||trim($opt['recommendation'])==='')$errors[]="$p.recommendation invalid";
                 if(isset($opt['link'])&&!is_string($opt['link']))$errors[]="$p.link invalid";}
        }
    } elseif ($hasR) {
        if(!is_string($node['recommendation'])||trim($node['recommendation'])==='')$errors[]="$path.recommendation invalid";
        if(isset($node['link'])&&!is_string($node['link']))$errors[]="$path.link invalid";
    } else $errors[]="$path must have question or recommendation";
}

add_action('admin_menu', function(){
    add_menu_page('Decision Trees v10','Decision Trees v10','manage_options','decision-tree-v10-admin','dtv10_render_admin','dashicons-feedback',20);
});

function dtv10_render_admin(){
    if(!current_user_can('manage_options')) wp_die('No permission');
    $dir=plugin_dir_path(__FILE__).'trees/'; if(!file_exists($dir))wp_mkdir_p($dir);

    if(isset($_POST['dt_tree_nonce'])&&wp_verify_nonce($_POST['dt_tree_nonce'],'dtv10_save_tree')){
        $name=sanitize_file_name($_POST['tree_name']); $content=wp_unslash($_POST['tree_content']);
        if($name===''){echo'<div class="notice notice-error"><p>Name required</p></div>';}
        else{$decoded=json_decode($content,true);
            if($decoded===null&&json_last_error()!==JSON_ERROR_NONE){
                echo'<div class="notice notice-error"><p>Invalid JSON</p></div>';
                dtv10_render_form($name,$content);return;
            }
            $errs=[];dtv10_validate_tree_array($decoded,$errs,'root');
            if($errs){echo'<div class="notice notice-error"><ul>';foreach($errs as $e)echo'<li>'.esc_html($e).'</li>';echo'</ul></div>';
                dtv10_render_form($name,$content);return;}
            $pretty=wp_json_encode($decoded,JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
            file_put_contents($dir.$name.'.json',$pretty);
            echo'<div class="notice notice-success"><p>Saved '.esc_html($name).'</p></div>';
        }
    }
    if(isset($_GET['delete'],$_GET['_wpnonce'])){
        $del=sanitize_file_name($_GET['delete']);
        if(wp_verify_nonce($_GET['_wpnonce'],'dtv10_delete_tree_'.$del)){
            $file=$dir.$del.'.json'; if(file_exists($file)) unlink($file);
            echo'<div class="notice notice-success"><p>Deleted '.esc_html($del).'</p></div>';
        }
    }
    if(isset($_GET['edit'])){$edit=sanitize_file_name($_GET['edit']);$file=$dir.$edit.'.json';
        $content=file_exists($file)?file_get_contents($file):'{}';dtv10_render_form($edit,$content);return;}
    if(isset($_GET['add'])){dtv10_render_form('',"{\n  \"question\": \"New question\",\n  \"options\": []\n}");return;}

    $trees=glob($dir.'*.json');
    echo'<div class="wrap"><h1>Decision Trees v10</h1>';
    echo'<table class="widefat"><thead><tr><th>Name</th><th>Shortcode</th><th>Actions</th></tr></thead><tbody>';
    if($trees){foreach($trees as $f){$n=basename($f,'.json');
        $edit=admin_url('admin.php?page=decision-tree-v10-admin&edit='.urlencode($n));
        $del=wp_nonce_url(admin_url('admin.php?page=decision-tree-v10-admin&delete='.urlencode($n)),'dtv10_delete_tree_'.$n);
        echo'<tr><td>'.esc_html($n).'</td><td>[decision_tree_v10 tree="'.esc_html($n).'"]</td><td><a href="'.$edit.'">Edit</a> | <a href="'.$del.'">Delete</a></td></tr>';
    }} else echo'<tr><td colspan=3>No trees</td></tr>';
    echo'</tbody></table>';
    $add=admin_url('admin.php?page=decision-tree-v10-admin&add=1');
    echo'<p><a class="button button-primary" href="'.$add.'">Add New</a></p></div>';
}

function dtv10_render_form($name,$content){
    echo'<div class="wrap"><h1>'.($name?'Edit '.esc_html($name):'Add New Tree').'</h1><form method="post">';
    wp_nonce_field('dtv10_save_tree','dt_tree_nonce');
    if($name)echo'<input type="hidden" name="tree_name" value="'.esc_attr($name).'">';
    else echo'<p>Name: <input type="text" name="tree_name" required></p>';
    echo'<textarea name="tree_content" rows="22" style="width:100%">'.esc_textarea($content).'</textarea>';
    echo'<div id="dt-json-errors" class="notice" style="display:none;margin-top:10px;"></div>';
    echo'<p><input type="submit" class="button button-primary" value="Save Tree"></p></form></div>';
}
